######################################################################################
## HPD plotting function
##     Rick Lewis May 2009
## 
## Function to make nice HPD interval plots with labeled groups. Assumes a
## data structure 'hpd.intervals' with the following columns:
##       coefficient   M  ci.lower   ci.upper  measure
##
## The two key parameters are "coef.groups",which defines the set of
## coefficients to plot, broken into groups, and "measures" which defines
## the list of measures to plot for each coefficient.
## Here's an example call:
##
##   plot.hpds(hpd, coef.groups = list(Group1=c("c1","c2","c3"),
##                                     Group2=c("c8","c9")),
##               measures = c("SPR", "TFT"),
##               group.labels = c(Group1="First Group", Group2= "Second Group"),
##               coef.labels = c(c1="Coef 1", c2="Coef 2", c3="Coef 3",
##                               c8="Coef 8", c9= "Coef 9")
##
##  It is highly customizable in terms of appearance of lines and it is
##  possible to distinguish significant vs. non-significant intervals.

plot.hpds <- function (hpd.intervals,
                       coef.groups,
                       measures,
                       coef.labels=NULL,
                       group.labels=NULL,
                       group.y.positioning=0.135,
                       cex.group.label=1,
                       ylim,
                       title="Title goes here",
                       subtitle="",
                       ylab="Coefficient estimate",
                       zero.lwd=0.5,
                       legend=TRUE,
                       x.axis=TRUE,
                       x.tick=TRUE,
                       measure.sep=0.3/(length(measures)-1),
                       x.pad = 0.4,
                       cex.axis=1.0,
                       cex.main=1.0,
                       cex.sub=1.0,
                       cex.lab=1.0,
                       cex.point=0.8,
                       measure.col = rep("black",length(measures)),
                       measure.lwd = rep(1.5, length(measures)),
                       measure.lty = rep(1, length(measures)),
                       measure.pch = rep(19, length(measures)),
                       non.sig.lwd= rep(1.5, length(measures)),
                       non.sig.lty= rep(3, length(measures)),
                       non.sig.col=NA,
                       non.sig.pch=NA) {

  ## Set up the plot
  coefs <- unlist(coef.groups);
  num.coef <- length(coefs);
  num.measures <- length(measures);
  xlim <- c(1-x.pad,num.coef+x.pad);
  par(bty="n", xpd=NA);  
  par(mgp=c(2.8,1,0));
  par(las=1);

  plot(x=0,y=0,
       type="n",
       bty="n",
       xaxt="n",
       xlab="",
       xlim=xlim,
       ylim=ylim,
       ylab=ylab,
       main=title,
       cex.main=cex.main,
       sub=subtitle,
       cex.sub=cex.sub,
       cex.lab=cex.lab,
       pch=19)

  ## Plot a zero baseline if requested
  if (zero.lwd)  lines(x=xlim, y=c(0,0), lwd=zero.lwd);

  ## Get set up for plotting the intervals. xdistrib is the set of
  ## num.measure x coordinates for each coeficient; determines their
  ## spacing based on the measure.sep paramter.
  xdistrib <- seq(-(num.measures - 1)/2 * measure.sep, (num.measures - 1)/2 * measure.sep, measure.sep)

  ## Loop through each coeficient
  for (c in 1:length(coefs)) {
    coef <- coefs[c];
    ## For each coeficient, plot each measure separately, choosing the
    ## appropriate line width, color etc. based on significance and
    ## specified parameters.
    for (m in 1:num.measures) {
      measure <- measures[m]
      x <- c + xdistrib[m]
      interval <- subset(hpd.intervals, coefficient==coef & measure==measures[m])
      ## Check for significance
      significant <-  (interval$ci.lower * interval$ci.upper > 0);

      ## Now draw the HPD interval
      lines(x=c(x,x),
            y=c(interval$ci.lower,interval$ci.upper),
            lwd=ifelse(significant | is.na(non.sig.lwd[m]), measure.lwd[m], non.sig.lwd[m]),
            lty=ifelse(significant | is.na(non.sig.lty[m]), measure.lty[m], non.sig.lty[m]),
            col=ifelse(significant | is.na(non.sig.col[m]), measure.col[m], non.sig.col[m]));
      
      ## And the point estimate
      points(x=x, y=interval$M, cex=cex.point,
             pch=ifelse(significant | is.na(non.sig.pch[m]), measure.pch[m], non.sig.pch[m]),
             col=ifelse(significant | is.na(non.sig.col[m]), measure.col[m], non.sig.col[m]))
      
    }
  }

  ## This draws the x axis and labels in the specified groups, using the
  ## alternative labels (if given).
  if (x.axis) {  ## only if requested
    x <- 1;
    ## For fine vertical positioning of the group label, see above
    group.label.y <- ylim[1] - diff(ylim)*group.y.positioning; 
    if (length(group.labels))
        glabels <-  group.labels[attr(coef.groups,"names")]
         else glabels <-  attr(coef.groups,"names");
    
    ## Loop through each group separately to make disjoint axis lines/ticks
    for (g in 1:length(coef.groups)) {
      cg <- coef.groups[[g]]
      if (length(coef.labels))   labels <- coef.labels[cg] else  labels <-cg;
      xrange <- x:(x+length(cg)-1);
      axis(1, at=xrange, labels=labels,cex.axis=cex.axis, tick=x.tick);
      ## Now add the group label
      group.label <- glabels[g];
      text(label=group.label,  x=mean(xrange), y = group.label.y,
           cex=cex.group.label)
      x <- x + length(cg);
    }
  }
}

