# Lena Jäger, Felix Engelmann, and Shravan Vasishth. 
# "Retrieval interference in reflexive processing: Experimental evidence from Mandarin, and computational modeling." 
# published in Frontiers in Psychology 6(617), 2015.

# Data analysis of Experiment 2


# Design: 2x3 factorial design:
# Locality (local antecedent; non-local antecedent)
# Animacy of the three memory load words (animate nouns; inanimate nouns; adjectives)
	# NOTE: In the Jäger, Engelmann, and Shravan Vasishth (2015) only the animacy contrast of the memory load words is reported. The comparisons involving adjectives as memory load are not reported in the paper. In this script, the whole data analysis is presented. 

# Description of the dataset:
# item: Item Id
# condition: experimental condition; coded as follows: 
	# a: local antecedent; three animate nouns as memory load
	# b: local antecedent; three inanimate nouns as memory load
	# c: local antecedent; three adjectives as memory load
	# d: non-local antecedent; three animate nouns as memory load
	# e: non-local antecedent; three inanimate nouns as memory load
	# f: non-local antecedent; three adjectives as memory load
# subject: participantId
# item: ItemId
# session: experimental session (2 sessions in total)
# accuracy: accuracy of the participant's response to the comprehension question;
	# coded as 1 if the response is correct, 0 otherwise
#  correct_answer: correct answer ('yes' or 'no') to the comprehension question 
# roi: region of interest coded as follows:
	# 1: NP1 (main clause subject including determiner+classifier if applicable)
	# 2: V1 (main clause verb)
	# 3: NP2 (embedded subject including determiner+classifier if applicable)
	# 4: pre-critical region (V2, embedded verb including aspect particle if applicable)
	# 5: critical region (ziji, 2 characters)
	# 6: post-critical region (durational phrase/ frequency phrase consisting of 4 characters)	
# FFD: first-fixation duration
# FFP: first-fixation probability coded as 1 if the region was fixated in the first pass, 0 otherwise
# SFD: single-fixation duration
# FPRT: first-pass reading time
# RBRT: right-bounded reading time
# TFT: total-fixation time
# RPD: regression-path duration
# RRT: re-reading time
# RegP: first-pass regression probability coded as 1 if a regression was initiated in the first pass from the respective region, 0 otherwise
# RRProb: Re-reaing probability coded as 1 if the region was re-read, 0 otherwise
# locality: locality of the reflexive-antecedent dependency (local vs non-local)
# memory: type of the memory load words (animte nouns, inanimate nouns, adjectives)	
# freq_NP1: frequency of the main clause subject (tokens per mio words)
# freq_NP2: frequency of the embedded subject (tokens per mio words)
	# frequencies have been extracted from the SUBLETEX-CH database (Cai and Brysbaert, 2010)


library(MASS)
library(lme4)

rm(list=ls())

data<-read.table("JaegerEngelmannVasishth2015Exp2.txt", header=T)

# column for frequency of the reflexive's antecedent
data$freq_Ant<-ifelse(data$locality=="local", data$freq_NP2, data$freq_NP1)
# center frequencies (z-scores):
data$freq_Ant<- (data$freq_Ant-mean(data$freq_Ant, na.rm=T))/sd(data$freq_Ant, na.rm=T)


data$subject<-factor(data$subject)
data$item<-factor(data$item)
data$session<-factor(data$session)
data$question_type <-factor(data$question_type)

#------------------------------ CONTRAST CODING ---------------------------------------------

contrasts(data$session)<-contr.sum(2)

 
# contrast matrix 1: ME of locality, animate-inanimate, inanimate-adj, (animate-inanimate)xlocality, (inanimate-adj)xlocality
cmat.1<-matrix(c(-1/3,-1/3,-1/3,1/3,1/3,1/3,1/2,-1/2,0,1/2,-1/2,0,0,1/2,-1/2,0,1/2,-1/2,1/2,-1/2,0,-1/2,1/2,0,0,1/2,-1/2,0,-1/2,1/2),6,5,byrow=F)
rownames(cmat.1) <- c("a", "b", "c", "d", "e", "f")
colnames(cmat.1) <- c("Locality", "Interference", "inan-adj", "Interference * Locality", "(inan-adj) * Loc")
(cmat.1<-fractions(cmat.1))
inv.cmat.1 <- t(ginv(cmat.1))
rownames(inv.cmat.1) <- c("a", "b", "c", "d", "e", "f")
colnames(inv.cmat.1) <- c("Locality", "Interference", "inan-adj", "Interference * Locality", "(inan-adj) * Loc")
(inv.cmat.1<-fractions(inv.cmat.1))

# contrast matrix 2: ME of locality; nested contrasts: a-b, b-c, d-e, e-f
cmat.2<-matrix(c(-1/3,-1/3,-1/3,1/3,1/3,1/3,1,-1,0,0,0,0,0,1,-1,0,0,0,0,0,0,1,-1,0,0,0,0,0,1,-1), 6, 5, byrow=F)
rownames(cmat.2) <- c("a", "b", "c", "d", "e", "f")
colnames(cmat.2) <- c("Locality", "Interference_local", "b-c", "Interference_nonlocal", "e-f")
(cmat.2<-fractions(cmat.2))
inv.cmat.2 <- t(ginv(cmat.2))
rownames(inv.cmat.2) <- c("a", "b", "c", "d", "e", "f")
colnames(inv.cmat.2) <- c("Locality", "Interference_local", "b-c", "Interference_nonlocal", "e-f")
(inv.cmat.2<-fractions(inv.cmat.2))

# ------------------add columns in model matrix for contrast coding -----------------
# contrasts equivalent to inv.cmat.1:
# main effect of Locality (d+e+f)-(a+b+c); nonlocal: 1; local: -1
data$Locality<-ifelse(data$locality=="local",-0.5, 0.5)
# main effect of Interference: animate-inanimate conditions (a+d) - (b+e)
data$Interference<-ifelse(data$condition=="a" | data$condition == "d",2/3, -1/3)
# inanimate-adjectives conditions (b+e) - (c+f)
data$inan_adj<-ifelse(data$condition=="a" | data$condition == "b"| data$condition == "d" | data$condition == "e",1/3, -2/3)
# Interference x Locality
data$Int_x_Loc<-ifelse(data$condition=="a",2/3, ifelse(data$condition=="b" | data$condition == "c",-1/3,ifelse(data$condition=="d",-2/3,1/3)))
# inan_adj x locality
data$inanadj_x_loc<-ifelse(data$condition=="a" | data$condition=="b",1/3, ifelse(data$condition == "c",-2/3,ifelse(data$condition=="f",2/3,-1/3)))

# contrasts equivalent to inv.cmat.2:
# ME of locality: specified above
# local_animate - local_inanimate: a-b
data$Interference_local<-ifelse(data$condition=="a" ,2/3, ifelse(data$condition=="b" |data$condition=="c",-1/3,0))
# local_inanimate - local_adjectives: b-c
data$b_c<- ifelse(data$condition=="a" |data$condition=="b" ,1/3, ifelse(data$condition=="c",-2/3,0))
# nonlocal_animate - nonlocal_inanimate: d-e
data$Interference_nonlocal<-ifelse(data$condition=="d" ,2/3, ifelse(data$condition=="e" |data$condition=="f",-1/3,0))
# nonlocal_inanimate - nonlocal_adjectives: e-f
data$e_f<- ifelse(data$condition=="d" | data$condition =="e" ,1/3, ifelse(data$condition=="f",-2/3,0))



# ------------------- Subset data for critical, post-critical and pre-critical regions ----------------------------
# critical region: 
dataziji<-subset(data, roi==5) 
# post-critical region: 
dataspill<-subset(data, roi==6)
# pre-critical region: 
dataverb<-subset(data,roi==4)




# ----------------------------------------------------------------------------------------------
#                    ANALYSIS OF RESPONSE ACCURACIES TO THE COMPREHENSION QUESTIONS 
# ----------------------------------------------------------------------------------------------
round(mean(dataziji$accuracy, na.rm=T), 2)
# Response accuracy by experimental condition:
round(tapply(dataziji$accuracy,list(dataziji$locality, dataziji$memory),mean, na.rm=T),3)
# Response accuracy by by correct answer (yes/no):
round(tapply(dataziji$accuracy, dataziji$correct_answer, mean, na.rm=T),3)

# accuracy by correct answer and by experimental condition
round(tapply(dataziji$accuracy, list(dataziji$locality, dataziji$memory, dataziji$correct_answer), mean, na.rm=T),2)

acc1<-glmer(accuracy~session*(Locality+Interference+inan_adj+Int_x_Loc+inanadj_x_loc)+(1|subject)+(1|item), data=dataziji, family=binomial(), control=glmerControl(optimizer="bobyqa"))
summary(acc1)
	# NOTE: The effect of Locality reaches statistical significance at an alpha level of 0.05 when using lme4 package version 1.1-9. In the original data analysis conducted with lme4 version 1.0-6, which is reported in Jäger, Engelmann, and Vasishth (2015), this effect was not significant. 



acc2<-glmer(accuracy~session*(Locality+Interference_local+b_c+Interference_nonlocal+e_f)+(1|subject)+(1|item), data=dataziji, family=binomial(), control=glmerControl(optimizer="bobyqa"))
summary(acc2)



# ----------------------------------------------------------------------------------------------
#                                   ANALYSIS OF EYE MOVEMENTS
# ----------------------------------------------------------------------------------------------     

#### First-fixation probability:
# pre-critical region: 
round(mean(dataverb$FFP),2)
# critical region: 
round(mean(dataziji$FFP),2)
# post-critical region: 
round(mean(dataspill$FFP),2)

#### Re-reading probability:
# pre-critical region: 
round(mean(dataverb$RRProb),2)
# critical region: 
round(mean(dataziji$RRProb),2)
# post-critical region: 
round(mean(dataspill$RRProb),2)

#-------------------- Subset data for Reading Measures  ----------------
#region: critical (ziji)
firstfix<-subset(dataziji, FFD!=0)
firstpass<-subset(dataziji,FPRT!=0 & FFP==1)
rightbound<-subset(dataziji,RBRT!=0)
totalfix<-subset(dataziji,TFT!=0)
regpath<-subset(dataziji,RPD!=0)
reread<-subset(dataziji,RRT!=0)

#region: post-critical
firstfixspill<-subset(dataspill, FFD!=0)
firstpassspill<-subset(dataspill,FPRT!=0 & FFP==1)
rightboundspill<-subset(dataspill,RBRT!=0)
totalfixspill<-subset(dataspill,TFT!=0)
regpathspill<-subset(dataspill,RPD!=0)
rereadspill<-subset(dataspill,RRT!=0)

#region: pre-critical 
firstfixverb<-subset(dataverb, FFD!=0)
firstpassverb<-subset(dataverb,FPRT!=0 & FFP==1)
rightboundverb<-subset(dataverb,RBRT!=0)
totalfixverb<-subset(dataverb,TFT!=0)
regpathverb<-subset(dataverb,RPD!=0)
rereadverb<-subset(dataverb,RRT!=0)



# --------------- Choose appropirate tranformation of the DVs --------------------------------
#region: critcal (ziji)
boxcox(FFD~subject, data=firstfix)
boxcox(FPRT~subject, data=firstpass)
boxcox(RBRT~subject, data=rightbound)
boxcox(TFT~subject, data=totalfix)
boxcox(RPD~subject, data=regpath)
boxcox(RRT~subject, data=reread)

#=> lambda is close to 0 in (almost) all measures => use log(DV)



# ----------------------------------------------------------------------------------------------
#                          Eye movements at the critical region (ziji, consists of 2 characters)
# ----------------------------------------------------------------------------------------------

# ------------------------------------------------------ FFD  ------------------------------------------------------ 
z1ffd<-lmer(log(FFD)~session*(Locality+Interference+inan_adj+Int_x_Loc+inanadj_x_loc+(1|subject)+(1|item)), data=firstfix, control=lmerControl(optimizer="bobyqa"))
summary(z1ffd)

z2ffd<-lmer(log(FFD)~session*(Locality+Interference_local+b_c+Interference_nonlocal+e_f+(1|subject)+(1|item)), data=firstfix, control=lmerControl(optimizer="bobyqa"))
summary(z2ffd)
 
# ------------------------------------------------------ FPRT ------------------------------------------------------ 
z1fprt<-lmer(log(FPRT)~session*(Locality+Interference+inan_adj+Int_x_Loc+inanadj_x_loc)+(1|subject)+(1|item), data=firstpass,  control=lmerControl(optimizer="bobyqa"))
summary(z1fprt)

z2fprt<-lmer(log(FPRT)~session*(Locality+Interference_local+b_c+Interference_nonlocal+e_f)+(1|subject)+(1|item), data=firstpass,  control=lmerControl(optimizer="bobyqa"))
summary(z2fprt)

# ------------------------------------------------------ RRT ------------------------------------------------------ 
z1rrt<-lmer(log(RRT)~session*(Locality+Interference+inan_adj+Int_x_Loc+inanadj_x_loc)+(1|subject)+(1|item), data=reread, control=lmerControl(optimizer="bobyqa"))
summary(z1rrt)

z2rrt<-lmer(log(RRT)~session*(Locality+Interference_local+b_c+Interference_nonlocal+e_f)+(1|subject)+(1|item), data=reread, control=lmerControl(optimizer="bobyqa"))
summary(z2rrt)

# ------------------------------------------------------ RBRT ------------------------------------------------------ 
z1rbrt<-lmer(log(RBRT)~session*(Locality+Interference+inan_adj+Int_x_Loc+inanadj_x_loc+(1|subject)+(1|item)), data=rightbound, control=lmerControl(optimizer="bobyqa"))
summary(z1rbrt)

z2rbrt<-lmer(log(RBRT)~session*(Locality+Interference_local+b_c+Interference_nonlocal+e_f)+(1|subject)+(1|item), data=rightbound, control=lmerControl(optimizer="bobyqa"))
summary(z2rbrt)

# ------------------------------------------------------ TFT ------------------------------------------------------  
z1tft<-lmer(log(TFT)~session*(Locality+Interference+inan_adj+Int_x_Loc+inanadj_x_loc)+(1|subject)+(1|item), data=totalfix, control=lmerControl(optimizer="bobyqa"))
summary(z1tft)

z2tft<-lmer(log(TFT)~session*(Locality+Interference_local+b_c+Interference_nonlocal+e_f)+(1|subject)+(1|item), data=totalfix, control=lmerControl(optimizer="bobyqa"))
summary(z2tft)

# ------------------------------------------------------ RPD ------------------------------------------------------
z1rpd<-lmer(log(RPD)~session*(Locality+Interference+inan_adj+Int_x_Loc+inanadj_x_loc)+(1|subject)+(1|item), data=regpath, control=lmerControl(optimizer="bobyqa"))
summary(z1rpd)

z2rpd<-lmer(log(RPD)~session*(Locality+Interference_local+b_c+Interference_nonlocal+e_f)+(1|subject)+(1|item), data=regpath, control=lmerControl(optimizer="bobyqa"))
summary(z2rpd)

# ------------------------------------------------------ First-pass regression probability ------------------------------------------------------
z1regp<-glmer(RegP~session*(Locality+Interference+inan_adj+Int_x_Loc+inanadj_x_loc)+(1|subject)+(1|item), data=dataziji, family=binomial(), glmerControl(optimizer="bobyqa"))
summary(z1regp)

z2regp<-glmer(RegP~session*(Locality+Interference_local+b_c+Interference_nonlocal+e_f)+(1|subject)+(1|item), data=dataziji, family=binomial(), glmerControl(optimizer="bobyqa"))
summary(z2regp)


# -----------------------------------------------------------------------------------------------------
#            Eye movements at the pre-critical region (verb including aspect particle if applicable)
# -----------------------------------------------------------------------------------------------------

# ------------------------------------------------------ FFD  ------------------------------------------------------ 
v1ffd<-lmer(log(FFD)~session*(Locality+Interference+inan_adj+Int_x_Loc+inanadj_x_loc+(1|subject)+(1|item)), data=firstfixverb, control=lmerControl(optimizer="bobyqa"))
summary(v1ffd)

v2ffd<-lmer(log(FFD)~session*(Locality+Interference_local+b_c+Interference_nonlocal+e_f+(1|subject)+(1|item)), data=firstfixverb, control=lmerControl(optimizer="bobyqa"))
summary(v2ffd)

# ------------------------------------------------------ FPRT ------------------------------------------------------ 
v1fprt<-lmer(log(FPRT)~session*(Locality+Interference+inan_adj+Int_x_Loc+inanadj_x_loc)+(1|subject)+(1|item), data=firstpassverb,  control=lmerControl(optimizer="bobyqa"))
summary(v1fprt)

v2fprt<-lmer(log(FPRT)~session*(Locality+Interference_local+b_c+Interference_nonlocal+e_f)+(1|subject)+(1|item), data=firstpassverb,  control=lmerControl(optimizer="bobyqa"))
summary(v2fprt)

# ------------------------------------------------------ RRT ------------------------------------------------------ 
v1rrt<-lmer(log(RRT)~session*(Locality+Interference+inan_adj+Int_x_Loc+inanadj_x_loc)+(1|subject)+(1|item), data=rereadverb, control=lmerControl(optimizer="bobyqa"))
summary(v1rrt)

v2rrt<-lmer(log(RRT)~session*(Locality+Interference_local+b_c+Interference_nonlocal+e_f)+(1|subject)+(1|item), data=rereadverb, control=lmerControl(optimizer="bobyqa"))
summary(v2rrt)

# ------------------------------------------------------ RBRT ------------------------------------------------------ 
v1rbrt<-lmer(log(RBRT)~session*(Locality+Interference+inan_adj+Int_x_Loc+inanadj_x_loc+(1|subject)+(1|item)), data=rightboundverb, control=lmerControl(optimizer="bobyqa"))
summary(v1rbrt)

v2rbrt<-lmer(log(RBRT)~session*(Locality+Interference_local+b_c+Interference_nonlocal+e_f)+(1|subject)+(1|item), data=rightboundverb, control=lmerControl(optimizer="bobyqa"))
summary(v2rbrt)

# ------------------------------------------------------ TFT ------------------------------------------------------ 
v1tft<-lmer(log(TFT)~session*(Locality+Interference+inan_adj+Int_x_Loc+inanadj_x_loc)+(1|subject)+(1|item), data=totalfixverb, control=lmerControl(optimizer="bobyqa"))
summary(v1tft)

v2tft<-lmer(log(TFT)~session*(Locality+Interference_local+b_c+Interference_nonlocal+e_f)+(1|subject)+(1|item), data=totalfixverb, control=lmerControl(optimizer="bobyqa"))
summary(v2tft)

# ------------------------------------------------------ RPD ------------------------------------------------------
v1rpd<-lmer(log(RPD)~session*(Locality+Interference+inan_adj+Int_x_Loc+inanadj_x_loc)+(1|subject)+(1|item), data=regpathverb, control=lmerControl(optimizer="bobyqa"))
summary(v1rpd)

v2rpd<-lmer(log(RPD)~session*(Locality+Interference_local+b_c+Interference_nonlocal+e_f)+(1|subject)+(1|item), data=regpathverb, control=lmerControl(optimizer="bobyqa"))
summary(v2rpd)

# ------------------------------------------------------ First-pass regression probability ------------------------------------------------------
v1regp<-glmer(RegP~session*(Locality+Interference+inan_adj+Int_x_Loc+inanadj_x_loc)+(1|subject)+(1|item), data=dataverb, family=binomial(), glmerControl(optimizer="bobyqa"))
summary(v1regp)

v2regp<-glmer(RegP~session*(Locality+Interference_local+b_c+Interference_nonlocal+e_f)+(1|subject)+(1|item), data=dataverb, family=binomial(), glmerControl(optimizer="bobyqa"))
summary(v2regp)


# ----------------------------------------------------------------------------------------------------------------------
#                     Eye movements at the post-critical region ("spillover" material consisting of 4 characters)
# ----------------------------------------------------------------------------------------------------------------------

# ------------------------------------------------------ FFD  ------------------------------------------------------ 
s1ffd<-lmer(log(FFD)~session*(Locality+Interference+inan_adj+Int_x_Loc+inanadj_x_loc+(1|subject)+(1|item)), data=firstfixspill, control=lmerControl(optimizer="bobyqa"))
summary(s1ffd)

s2ffd<-lmer(log(FFD)~session*(Locality+Interference_local+b_c+Interference_nonlocal+e_f+(1|subject)+(1|item)), data=firstfixspill, control=lmerControl(optimizer="bobyqa"))
summary(s2ffd)
 
# ------------------------------------------------------ FPRT ------------------------------------------------------ 
s1fprt<-lmer(log(FPRT)~session*(Locality+Interference+inan_adj+Int_x_Loc+inanadj_x_loc)+(1|subject)+(1|item), data=firstpassspill,  control=lmerControl(optimizer="bobyqa"))
summary(s1fprt)

s2fprt<-lmer(log(FPRT)~session*(Locality+Interference_local+b_c+Interference_nonlocal+e_f)+(1|subject)+(1|item), data=firstpassspill,  control=lmerControl(optimizer="bobyqa"))
summary(s2fprt)

# ------------------------------------------------------ RRT ------------------------------------------------------ 
s1rrt<-lmer(log(RRT)~session*(Locality+Interference+inan_adj+Int_x_Loc+inanadj_x_loc)+(1|subject)+(1|item), data=rereadspill, control=lmerControl(optimizer="bobyqa"))
summary(s1rrt)

s2rrt<-lmer(log(RRT)~session*(Locality+Interference_local+b_c+Interference_nonlocal+e_f)+(1|subject)+(1|item), data=rereadspill, control=lmerControl(optimizer="bobyqa"))
summary(s2rrt)

# ------------------------------------------------------ RBRT ------------------------------------------------------ 
s1rbrt<-lmer(log(RBRT)~session*(Locality+Interference+inan_adj+Int_x_Loc+inanadj_x_loc+(1|subject)+(1|item)), data=rightboundspill, control=lmerControl(optimizer="bobyqa"))
summary(s1rbrt)

s2rbrt<-lmer(log(RBRT)~session*(Locality+Interference_local+b_c+Interference_nonlocal+e_f)+(1|subject)+(1|item), data=rightboundspill, control=lmerControl(optimizer="bobyqa"))
summary(s2rbrt)

# ------------------------------------------------------ TFT ------------------------------------------------------ 
s1tft<-lmer(log(TFT)~session*(Locality+Interference+inan_adj+Int_x_Loc+inanadj_x_loc)+(1|subject)+(1|item), data=totalfixspill, control=lmerControl(optimizer="bobyqa"))
summary(s1tft)

s2tft<-lmer(log(TFT)~session*(Locality+Interference_local+b_c+Interference_nonlocal+e_f)+(1|subject)+(1|item), data=totalfixspill, control=lmerControl(optimizer="bobyqa"))
summary(s2tft)

# ------------------------------------------------------ RPD ------------------------------------------------------
s1rpd<-lmer(log(RPD)~session*(Locality+Interference+inan_adj+Int_x_Loc+inanadj_x_loc)+(1|subject)+(1|item), data=regpathspill, control=lmerControl(optimizer="bobyqa"))
summary(s1rpd)

s2rpd<-lmer(log(RPD)~session*(Locality+Interference_local+b_c+Interference_nonlocal+e_f)+(1|subject)+(1|item), data=regpathspill, control=lmerControl(optimizer="bobyqa"))
summary(s2rpd)

# ------------------------------------------------------ First-pass Regression probability ------------------------------
s1regp<-glmer(RegP~session*(Locality+Interference+inan_adj+Int_x_Loc+inanadj_x_loc)+(1|subject)+(1|item), data=dataspill, family=binomial(), glmerControl(optimizer="bobyqa"))
summary(s1regp)

s2regp<-glmer(RegP~session*(Locality+Interference_local+b_c+Interference_nonlocal+e_f)+(1|subject)+(1|item), data=dataspill, family=binomial(), glmerControl(optimizer="bobyqa"))
summary(s2regp)




####################### TABLE OF MEANS WITH STANDARD ERRORS: #######################
# define function for standard error for continuous DVs or proportions 
# 		(here only used for regression probabilities; SEs of continuous variables are computed with Morey-factor)
compute_se <- function(x){
	if (length(unique(x[!is.na(x)])>2) ){
		n <- length(x[!is.na(x)])
		sd(x, na.rm=TRUE) / sqrt(n)
	} else {
		m <- mean(x, na.rm=TRUE)
		n <- length(x[!is.na(x)])
		sqrt((m * (1 - m)) / n)
	}
}

# Function for SE with Morey correction factor:
# DV: vector containing dependent variable
# mf: Moray factor
SE <- function(DV,  MF){MF*sd(DV)/sqrt(length(DV))}
# set Morey factor for the present experiment:
mf <- sqrt(4/3) # Morey factor; because number of within-subj conditions = 4 (conditions with adjectives as memory load are removed)



#### Tabble with condition means at ziji (critical region):
firstfix<-subset(dataziji, FFD!=0)
FFD<-t(round(tapply(firstfix$FFD,firstfix$condition,mean)))
FFD_SE<-t(round(tapply(firstfix$FFD, firstfix$condition, SE, mf)))
for (i in 1:6){ # for all conditions
	FFD[i]<-paste(FFD[i], FFD_SE[i], sep="(")
	FFD[i]<-paste(FFD[i], ")", sep="")
}
FFD<-FFD[c(1,2,4,5)] # exclude conditions with adjectives as memory load

firstpass<-subset(dataziji,FPRT!=0 & FFP==1)
FPRT<-t(round(tapply(firstpass$FPRT,firstpass$condition,mean)))
FPRT_SE<-t(round(tapply(firstpass$FPRT, firstpass$condition, SE, mf)))
for (i in 1:6){ # for all conditions
	FPRT[i]<-paste(FPRT[i], FPRT_SE[i], sep="(")
	FPRT[i]<-paste(FPRT[i], ")", sep="")
}
FPRT<-FPRT[c(1,2,4,5)]
RRProb<-t(round(tapply(dataziji$RRProb, dataziji$condition, mean),2))
RRProb_SE<-t(round(tapply(dataziji$RRProb, dataziji$condition, compute_se),2))
for (i in 1:6){ # for all conditions
	RRProb[i]<-paste(RRProb[i], RRProb_SE[i], sep="(")
	RRProb[i]<-paste(RRProb[i], ")", sep="")
}
RRProb<-RRProb[c(1,2,4,5)]
RRT<-t(round(tapply(reread$RRT,reread$condition,mean)))
RRT_SE<-t(round(tapply(reread$RRT, reread$condition, SE, mf)))
for (i in 1:6){ # for all conditions
	RRT[i]<-paste(RRT[i], RRT_SE[i], sep="(")
	RRT[i]<-paste(RRT[i], ")", sep="")
}
RRT<-RRT[c(1,2,4,5)]
RBRT<-t(round(tapply(rightbound$RBRT,rightbound$condition,mean)))
RBRT_SE<-t(round(tapply(rightbound$RBRT, rightbound$condition, SE, mf)))
for (i in 1:6){ # for all conditions
	RBRT[i]<-paste(RBRT[i], RBRT_SE[i], sep="(")
	RBRT[i]<-paste(RBRT[i], ")", sep="")
}
RBRT<-RBRT[c(1,2,4,5)]
TFT<-t(round(tapply(totalfix$TFT,totalfix$condition,mean)))
TFT_SE<-t(round(tapply(totalfix$TFT, totalfix$condition, SE, mf)))
for (i in 1:6){ # for all conditions
	TFT[i]<-paste(TFT[i], TFT_SE[i], sep="(")
	TFT[i]<-paste(TFT[i], ")", sep="")
}
TFT<-TFT[c(1,2,4,5)]
RPD<-t(round(tapply(regpath$RPD,regpath$condition,mean)))
RPD_SE<-t(round(tapply(regpath$RPD, regpath$condition, SE, mf)))
for (i in 1:6){ # for all conditions
	RPD[i]<-paste(RPD[i], RPD_SE[i], sep="(")
	RPD[i]<-paste(RPD[i], ")", sep="")
}
RPD<-RPD[c(1,2,4,5)]
RegP<-t(round(tapply(dataziji$RegP, dataziji$condition,mean),2))
RegP_SE<-t(round(tapply(dataziji$RegP, dataziji$condition, compute_se),2))
for (i in 1:6){ # for all conditions
	RegP[i]<-paste(RegP[i], RegP_SE[i], sep="(")
	RegP[i]<-paste(RegP[i], ")", sep="")
}
RegP<-RegP[c(1,2,4,5)]

meansZiji<-rbind(FFD, FPRT, RBRT,RPD,RegP,RRProb,RRT,TFT)
conditions<-c("local/animate","local/inanimate", "nonlocal/animate","nonlocal/inanimate")
colnames(meansZiji)<-conditions
print(meansZiji)

#------------------------------------------------------------------------------------------

# at verb (pre-critical region)
FFD<-t(round(tapply(firstfixverb$FFD,firstfixverb$condition,mean)))
FFD_SE<-t(round(tapply(firstfixverb$FFD, firstfixverb$condition, SE, mf)))
for (i in 1:6){ # for all conditions
	FFD[i]<-paste(FFD[i], FFD_SE[i], sep="(")
	FFD[i]<-paste(FFD[i], ")", sep="")
}
FPRT<-t(round(tapply(firstpassverb$FPRT,firstpassverb$condition,mean)))
FPRT_SE<-t(round(tapply(firstpassverb$FPRT, firstpassverb$condition, SE, mf)))
for (i in 1:6){ # for all conditions
	FPRT[i]<-paste(FPRT[i], FPRT_SE[i], sep="(")
	FPRT[i]<-paste(FPRT[i], ")", sep="")
}
RegP<-t(round(tapply(dataverb$RegP, dataverb$condition,mean),2))
RegP_SE<-t(round(tapply(dataverb$RegP, dataverb$condition, compute_se),2))
for (i in 1:6){ # for all conditions
	RegP[i]<-paste(RegP[i], RegP_SE[i], sep="(")
	RegP[i]<-paste(RegP[i], ")", sep="")
}
RPD<-t(round(tapply(regpathverb$RPD,regpathverb$condition,mean)))
RPD_SE<-t(round(tapply(regpathverb$RPD, regpathverb$condition, SE, mf)))
for (i in 1:6){ # for all conditions
	RPD[i]<-paste(RPD[i], RPD_SE[i], sep="(")
	RPD[i]<-paste(RPD[i], ")", sep="")
}
RBRT<-t(round(tapply(rightboundverb$RBRT,rightboundverb$condition,mean)))
RBRT_SE<-t(round(tapply(rightboundverb$RBRT, rightboundverb$condition, SE, mf)))
for (i in 1:6){ # for all conditions
	RBRT[i]<-paste(RBRT[i], RBRT_SE[i], sep="(")
	RBRT[i]<-paste(RBRT[i], ")", sep="")
}
TFT<-t(round(tapply(totalfixverb$TFT,totalfixverb$condition,mean)))
TFT_SE<-t(round(tapply(totalfixverb$TFT, totalfixverb$condition, SE, mf)))
for (i in 1:6){ # for all conditions
	TFT[i]<-paste(TFT[i], TFT_SE[i], sep="(")
	TFT[i]<-paste(TFT[i], ")", sep="")
}
RRT<-t(round(tapply(rereadverb$RRT,rereadverb$condition,mean)))
RRT_SE<-t(round(tapply(rereadverb$RRT, rereadverb$condition, SE, mf)))
for (i in 1:6){ # for all conditions
	RRT[i]<-paste(RRT[i], RRT_SE[i], sep="(")
	RRT[i]<-paste(RRT[i], ")", sep="")
}
RRProb<-t(round(tapply(dataverb$RRProb, dataverb$condition,mean),2))
RRProb_SE<-t(round(tapply(dataverb$RRProb, dataverb$condition, compute_se),2))
for (i in 1:6){ # for all conditions
	RRProb[i]<-paste(RRProb[i], RRProb_SE[i], sep="(")
	RRProb[i]<-paste(RRProb[i], ")", sep="")
}

FFD<-FFD[c(1,2,4,5)]
FPRT<-FPRT[c(1,2,4,5)]
RRProb<-RRProb[c(1,2,4,5)]
RRT<-RRT[c(1,2,4,5)]
RBRT<-RBRT[c(1,2,4,5)]
TFT<-TFT[c(1,2,4,5)]
RPD<-RPD[c(1,2,4,5)]
RegP<-RegP[c(1,2,4,5)]

meansVerb<-rbind(FFD, FPRT, RBRT,RPD,RegP,RRProb,RRT,TFT)
colnames(meansVerb)<-conditions
print(meansVerb)
tex.meansVerb <- tex.meansTable(meansVerb)

#------------------------------------------------------------------------------------------

# at spillover material (post-critical region)
FFD<-t(round(tapply(firstfixspill$FFD,firstfixspill$condition,mean)))
FFD_SE<-t(round(tapply(firstfixspill$FFD, firstfixspill$condition, SE, mf)))
for (i in 1:6){ # for all conditions
	FFD[i]<-paste(FFD[i], FFD_SE[i], sep="(")
	FFD[i]<-paste(FFD[i], ")", sep="")
}
FPRT<-t(round(tapply(firstpassspill$FPRT,firstpassspill$condition,mean)))
FPRT_SE<-t(round(tapply(firstpassspill$FPRT, firstpassspill$condition, SE, mf)))
for (i in 1:6){ # for all conditions
	FPRT[i]<-paste(FPRT[i], FPRT_SE[i], sep="(")
	FPRT[i]<-paste(FPRT[i], ")", sep="")
}
RegP<-t(round(tapply(dataspill$RegP, dataspill$condition,mean),2))
RegP_SE<-t(round(tapply(dataspill$RegP, dataspill$condition, compute_se),2))
for (i in 1:6){ # for all conditions
	RegP[i]<-paste(RegP[i], RegP_SE[i], sep="(")
	RegP[i]<-paste(RegP[i], ")", sep="")
}
RPD<-t(round(tapply(regpathspill$RPD,regpathspill$condition,mean)))
RPD_SE<-t(round(tapply(regpathspill$RPD, regpathspill$condition, SE, mf)))
for (i in 1:6){ # for all conditions
	RPD[i]<-paste(RPD[i], RPD_SE[i], sep="(")
	RPD[i]<-paste(RPD[i], ")", sep="")
}
RRT<-t(round(tapply(rereadspill$RRT,rereadspill$condition,mean)))
RRT_SE<-t(round(tapply(rereadspill$RRT, rereadspill$condition, SE, mf)))
for (i in 1:6){ # for all conditions
	RRT[i]<-paste(RRT[i], RRT_SE[i], sep="(")
	RRT[i]<-paste(RRT[i], ")", sep="")
}
TFT<-t(round(tapply(totalfixspill$TFT,totalfixspill$condition,mean)))
TFT_SE<-t(round(tapply(totalfixspill$TFT, totalfixspill$condition, SE, mf)))
for (i in 1:6){ # for all conditions
	TFT[i]<-paste(TFT[i], TFT_SE[i], sep="(")
	TFT[i]<-paste(TFT[i], ")", sep="")
}
RRProb<-t(round(tapply(dataspill$RRProb, dataspill$condition,mean),2))
RRProb_SE<-t(round(tapply(dataspill$RRProb, dataspill$condition, compute_se),2))
for (i in 1:6){ # for all conditions
	RRProb[i]<-paste(RRProb[i], RRProb_SE[i], sep="(")
	RRProb[i]<-paste(RRProb[i], ")", sep="")
}
RBRT<-t(round(tapply( rightboundspill$RBRT, rightboundspill$condition,mean))) 
RBRT_SE<-t(round(tapply(rightboundspill$RBRT, rightboundspill$condition, SE, mf)))
for (i in 1:6){ # for all conditions
	RBRT[i]<-paste(RBRT[i], RBRT_SE[i], sep="(")
	RBRT[i]<-paste(RBRT[i], ")", sep="")
}

FFD<-FFD[c(1,2,4,5)]
FPRT<-FPRT[c(1,2,4,5)]
RRProb<-RRProb[c(1,2,4,5)]
RRT<-RRT[c(1,2,4,5)]
RBRT<-RBRT[c(1,2,4,5)]
TFT<-TFT[c(1,2,4,5)]
RPD<-RPD[c(1,2,4,5)]
RegP<-RegP[c(1,2,4,5)]


meansSpill<-rbind(FFD, FPRT, RBRT,RPD,RegP,RRProb,RRT,TFT)
colnames(meansSpill)<-conditions
print(meansSpill)



