# Lena Jäger, Felix Engelmann, and Shravan Vasishth. 
# "Retrieval interference in reflexive processing: Experimental evidence from Mandarin, and computational modeling." 
# published in Frontiers in Psychology 6(617), 2015.

# Data analysis of Experiment 1


# Design: 2x2 factorial design:
# animacy of antecedent (animate/inanimate)
# animacy of distractor (animate/inanimate)

# Description of the dataset:
# condition: 
	# a) animate antecedent; animate distractor
	# b) animate antecedent; inanimate distractor
	# c) inanimate antecedent; animate distractor
	# d) inanimate antecedent; inanimate distractor

# subject: Participant Id
# Item: Item Id
# list: Latin Square List
# accuracy: accuracy of the answer to the multiple-choice comprehension question targeting the reflexive-antecedent dependency
	# coded as 1 if the correct answer was chosen, 0 otherwise.
	# the correct answer is always the antecedent.
# given_answer: the participant's answer to the  multiple-choice comprehension question targeting the reflexive-antecedent dependency which is coded as follows:
	# 1: antecedent
	# 2: distractor (inaccessible antecedent)
	# 3: unrelated noun (a word that did not occur in the sentence)
	# 4: "not sure"
# FFD: first-fixation duration
# FFP: first-fixation probability; coded as 1 if a region was fixated in the first-pass reading
# SFD: single-fixation duration (not analyzed)
# FPRT: first-pass reading time
# RBRT: right-bounded reading time
# TFT: total-fixation time
# RPD: regression-path duration
# RRT: re-reading time
# region: region of interest 
	# "ziji": the critical region containing the reflexive (2 characters)
	# "verb": the pre-critical region containing the main clause verb (including aspect particle if present) directly preceding "ziji" (2-3 characters)
	# "spillover": the post-critcal region containing the frequency/durational phrase directly following "ziji" (4 characters)
# Animacy_antecedent: animacy of the antecedent
# Animacy_intruder: animacy of the distractor (i.e., the inaccessible antecedent)
# RegP: first-pass regression probability;
	# coded as 1 if a regression was initiated from this region in the first-pass reading, 0 otherwise
# RRProb: re-reading probability; 
	# coded as 1 if the region was re-read, 0 otherwise
# freq_ant: frequency of the antecedent (per million words)
# freq_int:	frequency of the distractor, i.e., the inaccessible antecedent (per million words)
		# frequencies have been extracted from the SUBLETEX-CH database (Cai and Brysbaert, 2010)


rm(list=ls())

library(MASS)
library(lme4)
library(languageR)


data<-read.table("JaegerEngelmannVasishth2015Exp1.txt", header=T)
head(data)
# log transform frequencies of antecedent and intruder in order to get a close to uniform distribution
data$freq_ant<- log(data$freq_ant+1)
data$freq_int<- log(data$freq_int+1)

# center frequencies  (z-scores)
data$freq_ant<- (data$freq_ant-mean(data$freq_ant, na.rm=T))/sd(data$freq_ant, na.rm=T)
data$freq_int<- (data$freq_int-mean(data$freq_int, na.rm=T))/sd(data$freq_int, na.rm=T)

# ------------------------------------------------------------------------------------------
# Test for collinearity of frequency and animacy antecedent:
freq<-read.table("Frequencies.txt", header=T) # frequencies are words per mio.
colnames(freq)<-c("trial", "condition", "freq_ant", "freq_int")

freq$freq_ant<- log(freq$freq_ant+1)
freq$freq_int<- log(freq$freq_int+1)

# center frequencies 
freq$freq_ant<- (freq$freq_ant-mean(freq$freq_ant, na.rm=T))/sd(freq$freq_ant, na.rm=T)
freq$freq_int<- (freq$freq_int-mean(freq$freq_int, na.rm=T))/sd(freq$freq_int, na.rm=T)

freq$AnimacyAnt<-ifelse(freq$condition %in%c("a","b"), 1, 0)
freq$AnimacyInt<-ifelse(freq$condition %in%c("a","c"), 1, 0)
# exclude items with missing values for frequency of antecedent:
freqAccAnt<-subset(freq, !(is.na(freq_ant)))
# test collinearity of frequency and animacy of the accessible antecedent
collin.fnc(freqAccAnt, c(3,5))$cnumber  # this test follows Belsley et al 1980

# exclude items with missing values for frequency of antecedent:
freqAccInt<-subset(freq, !(is.na(freq_int)))
# test collinearity of frequency and animacy of the accessible antecedent
collin.fnc(freqAccInt, c(4,6))$cnumber  # this test follows Belsley et al 1980

# ------------------------------------------------------------------------------------------



#----------------------------------------------------------------------------------------------------------------------------------------------------
#				                                                  Contrast Coding
#----------------------------------------------------------------------------------------------------------------------------------------------------
#.....main effect of animacy of antecedent, main effect of animafy of the distractor (interference) and their interactions
cmat<-matrix(c(.5,-.5,.5,-.5,.5,.5,-.5,-.5,.5,-.5,-.5,.5),4,3)
colnames(cmat)<-c("ME Interference", "ME Antecedent", "Interference x Antecedent")
# ... compute the inverse
(inv.cmat <- t(ginv(cmat)))
rownames(inv.cmat) <- c("a", "b", "c", "d")
colnames(inv.cmat) <- c("Interference", "Antecedent", "Interference x Antecedent")
### matrix reported in the paper: same comparisons, but Animacy of Antecedent in first column, Animacy of Intruder in second column.


#.....nested contrasts: ME of antecedent; effect of intruder in grammatical and ungrammatical conditions separately
 cmat2<-matrix(c(1,-1,0,0,0,0,1,-1, .5, .5,-.5,-.5),4,3, byrow=FALSE)
colnames(cmat2)<-c("a-b", "c-d","MEant")
(inv.cmat2 <- fractions(t(ginv(cmat2))))
rownames(inv.cmat2) <- c("a", "b", "c", "d")
colnames(inv.cmat2) <- c("a-b", "c-d", "MEant")

# Add columns corresponding to the above specified contrasts:
data$MEant<-ifelse(data$condition%in%c("a","b"),0.5,-0.5) # main effect animacy of antecedent
data$MEint<-ifelse(data$condition%in%c("a","c"),0.5,-0.5) # main effect animacy of intruder
data$IntAnt<-ifelse(data$condition%in%c("a","d"),0.5,-0.5) # interaction of animacy of antecedent and animacy of intruder
data$gInt<-ifelse(data$condition=="a",0.5,ifelse(data$condition=="b", -0.5, 0)) # interference effect in grammatical conditions 
data$uInt<-ifelse(data$condition=="c",0.5,ifelse(data$condition=="d", -0.5, 0)) # interference effect in ungrammatical conditions


data$Item<-factor(data$Item)
data$subject<-factor(data$subject)
#########################################################################################

#----------------------------------------------------------------------------------------------------------------------------------------------------
#				                            Question 1: ACCURACIES and CHOSEN ANSWER
#----------------------------------------------------------------------------------------------------------------------------------------------------
# There are 4 multiple choice options (antecedent, distractor, unrelated noun, not sure) to chose from. 
# The correct answer is the antecedent. 
dataziji<-subset(data, region=="ziji")
xtabs(~condition+subject, dataziji) # for a few participants the experiment was aborted before all items had been seen for either technical reasons or the participant not feeling well


#----------------------------------------------------------------------------------------------------------------------------------------------------
#				                                                  Accuracies 
#----------------------------------------------------------------------------------------------------------------------------------------------------
(means<-round(tapply(dataziji$accuracy, dataziji$condition, mean, na.rm=T),3))

acc1<-glmer(accuracy~MEant+MEint+IntAnt+(1+MEant+MEint+IntAnt|subject)+(1+MEant+MEint+IntAnt|Item), data=dataziji, family=binomial(), control=glmerControl(optimizer="bobyqa"))
summary(acc1)  


#----------------------------------------------------------------------------------------------------------------------------------------------------
#			            								                    Given Answer
#----------------------------------------------------------------------------------------------------------------------------------------------------

# Was the DISTRACTOR (i.e., inaccessible antecedent) chosen more often in the interference conditions compared to the non-interference conditions?
# code answer_is_INT as 1 if inaccessible antecedent was chosen as answer, 0 otherwise
dataziji$answer_is_INT<-ifelse(dataziji$given_answer==2,1,ifelse(dataziji$given_answer%in%c(1,3,4),0,NA))

answInt1<-glmer(answer_is_INT~MEant+MEint+IntAnt+(1+MEant+MEint+IntAnt|subject)+(1+MEant+MEint+IntAnt|Item), data=dataziji, family=binomial(), control=glmerControl(optimizer="bobyqa"))
summary(answInt1)  

answInt2<-glmer(answer_is_INT~MEant+gInt+uInt+(1+MEant+gInt+uInt|subject)+(1+ MEant+gInt+uInt|Item), data=dataziji, family=binomial(), control=glmerControl(optimizer="bobyqa"))  
summary(answInt2)  





##########################################################################################

#----------------------------------------------------------------------------------------------------------------------------------------------------
#			            								                    EYE MOVEMENTS
#----------------------------------------------------------------------------------------------------------------------------------------------------
### All models  on continuous dependent variables are fit on log-transformed values;
### All models on binary dependent variables are fit with as a generalized linear mixed model with a logistic-link function.

### For all dependent variables / region of interest two sets of contrasts are fit (see contrast coding above):
### contrasts 1: main effect antecedent, main effect interference, interaction Int x Ant
### contrasts 2: nested comparisons: main effect antecedent, interference within antecedent-match a-b, interference within antecedent-mismatch c-d

# in addition to these contrasts, the log-frequency of the antecedent and the log-frequency of the distractor are included as predictors in all models. 

# random effects: item and subject are included as random effects (varying intercepts and slopes). 
# In case the varying slopes were removed in a model in order to avoid convergence failure, they are also removed from the corresponding model with the other set of contrasts.


# Labelling of lmer's: 
#.........z,s,v: codes region that is analyzed: 
			# z:=ziji (critical)
			# s:=spillover (post-critical)
			# v:=verb (pre-critical)
#.........1,2: codes applied contrasts: 
			# 1: main effect antecedent, main effect interference, interaction Int x Ant
			# 2: main effect antecedent, interference within antecedent-match a-b, interference within antecedent-mismatch c-d



# subset data for regions ziji, post-critical ("spillover") and pre-critical ("verb")
dataziji<-subset(data, region=="ziji")
dataspill<-subset(data, region=="spillover")
dataverb<-subset(data, region=="verb")


# --------------- Choose appropirate tranformation of the DVs --------------------------------
#region: ziji
zijifirstfix<-subset(dataziji, FFD!=0)
boxcox(FFD~subject, data=zijifirstfix)
zijifirstpass<-subset(dataziji,FPRT!=0 & FFP==1)
boxcox(FPRT~subject, data=zijifirstpass)
zijirbound<-subset(dataziji,RBRT!=0)
boxcox(RBRT~subject, data=zijirbound)
zijitotalfix<-subset(dataziji,TFT!=0)
boxcox(TFT~subject, data=zijitotalfix)
zijiregpath<-subset(dataziji,RPD!=0)
boxcox(RPD~subject, data=zijiregpath)
zijireread<-subset(dataziji,RRT!=0)
boxcox(RRT~subject, data=zijireread)

#region: post-critical
spillfirstfix<-subset(dataspill, FFD!=0)
boxcox(FFD~subject, data=spillfirstfix)
spillfirstpass<-subset(dataspill,FPRT!=0 & FFP==1)
boxcox(FPRT~subject, data=spillfirstpass)
spillrbound<-subset(dataspill,RBRT!=0)
boxcox(RBRT~subject, data=spillrbound)
spilltotalfix<-subset(dataspill,TFT!=0)
boxcox(TFT~subject, data=spilltotalfix)
spillregpath<-subset(dataspill,RPD!=0)
boxcox(RPD~subject, data=spillregpath)
spillreread<-subset(dataspill,RRT!=0)
boxcox(RRT~subject, data=spillreread)

# lambda is close to 0 in (almost) all measures => use log(DV)


#----------------------------------------------------------------------------------------------------------------------------------------------------

# First-fixation probability:
round(mean(dataverb$FFP),2)
round(mean(dataziji$FFP),2)
	# by condition:
	round((tapply(dataziji$FFP, dataziji$condition, mean)),2)
round(mean(dataspill$FFP),2)


# Re-reading probability:
round(mean(dataverb$RRProb),2)
round(mean(dataziji$RRProb),2)
	# by condition:
	round((tapply(dataziji$RRProb, dataziji$condition, mean)),2)
round(mean(dataspill$RRProb),2)





#----------------------------------------------------------------------------------------------------------------------------------------------------
#			            								                    Eye movements at "ziji"
#----------------------------------------------------------------------------------------------------------------------------------------------------

#-------------------------------------------------------------- FFD ------------------------------------------------------------------------------
zijifirstfix<-subset(dataziji, dataziji$FFD!=0)
z1ffd<-lmer(log(FFD)~freq_ant+freq_int+MEant+MEint+IntAnt+(1+MEant+MEint+IntAnt|subject)+(1+MEant+MEint+IntAnt|Item), data=zijifirstfix, control=lmerControl(optimizer="bobyqa"))  
summary(z1ffd)
z2ffd<-lmer(log(FFD)~freq_ant+freq_int+MEant+gInt+uInt+(1+MEant+gInt+uInt|subject)+(1+MEant+gInt+uInt|Item), data=zijifirstfix, control=lmerControl(optimizer="bobyqa"))
summary(z2ffd)

#-------------------------------------------------------------- FPRT ------------------------------------------------------------------------------
zijifirstpass<-subset(dataziji, dataziji$FPRT!=0 & FFP==1)
z1fprt<-lmer(log(FPRT)~freq_ant+freq_int+MEant+MEint+IntAnt+(1+MEant+MEint+IntAnt|subject)+(1+MEant+MEint+IntAnt|Item), data=zijifirstpass, control=lmerControl(optimizer="bobyqa"))  
summary(z1fprt)
z2fprt<-lmer(log(FPRT)~freq_ant+freq_int+MEant+gInt+uInt +(1+MEant+gInt+uInt|subject)+(1+MEant+uInt+gInt|Item), data=zijifirstpass, control=lmerControl(optimizer="bobyqa"))  
summary(z2fprt)

#-------------------------------------------------------------- RBRT ------------------------------------------------------------------------------
zijirbound<-subset(dataziji, dataziji$RBRT!=0)
z1rbrt<-lmer(log(RBRT)~freq_ant+freq_int+MEant+MEint+IntAnt+(1|subject)+(1+MEant+MEint+IntAnt|Item), data=zijirbound,  control=lmerControl(optimizer="bobyqa"))  
summary(z1rbrt)
z2rbrt<-lmer(log(RBRT)~freq_ant+freq_int+MEant+gInt+uInt+(1|subject)+(1+MEant+uInt+gInt|Item), data=zijirbound, control=lmerControl(optimizer="bobyqa"))  
summary(z2rbrt)

 #-------------------------------------------------------------- RPD ------------------------------------------------------------------------------
zijiregpath<-subset(dataziji, dataziji$RPD!=0)
z1rpd<-lmer(log(RPD)~freq_ant+freq_int+MEant+MEint+IntAnt+(1|subject)+(1|Item), data=zijiregpath, control=lmerControl(optimizer="bobyqa"))
summary(z1rpd)
(z2rpd<-lmer(log(RPD)~freq_ant+freq_int+MEant+gInt+uInt+(1|subject)+(1|Item), data=zijiregpath, control=lmerControl(optimizer="bobyqa")))    
summary(z2rpd)

#-------------------------------------------------------------- RRT ------------------------------------------------------------------------------
zijireread<-subset(dataziji, dataziji$RRT!=0)
z1rrt<-lmer(log(RRT)~freq_ant+freq_int+MEant+MEint+IntAnt+(1+MEant+MEint+IntAnt|subject)+(1|Item), data=zijireread, control=lmerControl(optimizer="bobyqa")) 
summary(z1rrt)
z2rrt<-lmer(log(RRT)~freq_ant+freq_int+MEant+gInt+uInt+(1+MEant+gInt+uInt|subject)+(1|Item), data=zijireread, control=lmerControl(optimizer="bobyqa"))
summary(z2rrt)

#-------------------------------------------------------------- TFT ------------------------------------------------------------------------------
zijitotalfix<-subset(dataziji, dataziji$TFT!=0)
z1tft<-lmer(log(TFT)~freq_ant+freq_int+MEant+MEint+IntAnt+(1+MEant+MEint+IntAnt|subject)+(1 |Item), data=zijitotalfix, control=lmerControl(optimizer="bobyqa"))  
summary(z1tft)
z2tft<-lmer(log(TFT)~freq_ant+freq_int+MEant+gInt+uInt+(1+MEant+gInt+uInt|subject)+(1|Item), data=zijitotalfix,  control=lmerControl(optimizer="bobyqa"))
summary(z2tft)

#-------------------------------------------------------------- FPRP ------------------------------------------------------------------------------
z1regp<-glmer(RegP~freq_ant+freq_int+MEant+MEint+IntAnt+(1|subject)+(1|Item), data=dataziji, family=binomial(), control=glmerControl(optimizer="bobyqa"))  
summary(z1regp)
z2regp<-glmer(RegP~freq_ant+freq_int+MEant+gInt+uInt+(1|subject)+(1|Item), data=dataziji, family=binomial(), control=glmerControl(optimizer="bobyqa"))  
summary(z2regp)



#----------------------------------------------------------------------------------------------------------------------------------------------------
#			            					       Eye movements at the post-critical region ("spillover") (4 characters)
#------------------------------------------------------------------------------------------------------------------------------------------------#-------------------------------------------------------------- FFD ------------------------------------------------------------------------------
spillfirstfix<-subset(dataspill, dataspill$FFD!=0)
s1ffd<-lmer(log(FFD)~freq_ant+freq_int+MEant+MEint+IntAnt+(1|subject)+(1|Item), data=spillfirstfix, control=lmerControl(optimizer="bobyqa"))  
summary(s1ffd)
s2ffd<-lmer(log(FFD)~freq_ant+freq_int+MEant+gInt+uInt+(1|subject)+(1|Item), data=spillfirstfix, control=lmerControl(optimizer="bobyqa")) 
summary(s2ffd)

#-------------------------------------------------------------- FPRT ------------------------------------------------------------------------------
spillfirstpass<-subset(dataspill, dataspill$FPRT!=0 & FFP==1)
s1fprt<-lmer(log(FPRT)~freq_ant+freq_int+MEant+MEint+IntAnt+(1|subject)+(1+MEant+MEint+IntAnt|Item), data=spillfirstpass, control=lmerControl(optimizer="bobyqa")) 
summary(s1fprt)
s2fprt<-lmer(log(FPRT)~freq_ant+freq_int+MEant+gInt+uInt+(1|subject)+(1+MEant+gInt+uInt|Item), data=spillfirstpass, control=lmerControl(optimizer="bobyqa"))
summary(s2fprt)

#-------------------------------------------------------------- RBRT ------------------------------------------------------------------------------
spillrbound<-subset(dataspill, dataspill$RBRT!=0)
s1rbrt<-lmer(log(RBRT)~freq_ant+freq_int+MEant+MEint+IntAnt+(1|subject)+(1|Item), data=spillrbound, control=lmerControl(optimizer="bobyqa")) 
summary(s1rbrt)
s2rbrt<-lmer(log(RBRT)~freq_ant+freq_int+MEant+gInt+uInt+(1|subject)+(1|Item), data=spillrbound, control=lmerControl(optimizer="bobyqa"))    
 summary(s2rbrt)

 #-------------------------------------------------------------- RPD ------------------------------------------------------------------------------
spillregpath<-subset(dataspill, dataspill$RPD!=0)
s1rpd<-lmer(log(RPD)~freq_ant+freq_int+MEant+MEint+IntAnt+(1+MEant+MEint+IntAnt|subject)+(1+MEant+MEint+IntAnt|Item), data=spillregpath,  control=lmerControl(optimizer="bobyqa"))
summary(s1rpd)
s2rpd<-lmer(log(RPD)~freq_ant+freq_int+MEant+gInt+uInt+(1+MEant+gInt+uInt|subject)+(1+MEant+gInt+uInt|Item), data=spillregpath,  control=lmerControl(optimizer="bobyqa"))    
summary(s2rpd)

#-------------------------------------------------------------- RRT ------------------------------------------------------------------------------
spillreread<-subset(dataspill, dataspill$RRT!=0)
s1rrt<-lmer(log(RRT)~freq_ant+freq_int+MEant+MEint+IntAnt+(1|subject)+(1|Item), data=spillreread,  control=lmerControl(optimizer="bobyqa")) 
summary(s1rrt)
s2rrt<-lmer(log(RRT)~freq_ant+freq_int+MEant+gInt+uInt+(1|subject)+(1|Item), data=spillreread,  control=lmerControl(optimizer="bobyqa"))
summary(s2rrt)


#-------------------------------------------------------------- TFT ------------------------------------------------------------------------------
spilltotalfix<-subset(dataspill, dataspill$TFT!=0)
s1tft<-lmer(log(TFT)~freq_ant+freq_int+MEant+MEint+IntAnt+(1+MEant+MEint+IntAnt|subject)+(1+MEant+MEint+IntAnt|Item), data=spilltotalfix, control=lmerControl(optimizer="bobyqa"))  
summary(s1tft)
s2tft<-lmer(log(TFT)~freq_ant+freq_int+MEant+gInt+uInt+(1+MEant+gInt+uInt|subject)+(1+MEant+gInt+uInt|Item), data=spilltotalfix, control=lmerControl(optimizer="bobyqa"))  
summary(s2tft)

#-------------------------------------------------------------- FPRP ------------------------------------------------------------------------------
s1regp<-glmer(RegP~freq_ant+freq_int+MEant+MEint+IntAnt+(1+MEant+MEint+IntAnt|subject)+(1|Item), data=dataspill, family=binomial(), control=glmerControl(optimizer="bobyqa"))  
summary(s1regp)
s2regp<-glmer(RegP~freq_ant+freq_int+MEant+gInt+uInt+(1+MEant+gInt+uInt|subject)+(1|Item), data=dataspill, family=binomial(), control=glmerControl(optimizer="bobyqa"))  
summary(s2regp)




#------------------------------------------------------------------------------------------------------------------------------------------------
#			           Eye movements at the pre-critical region (main clause verb region preceding ziji) (2-3 characters length)
#------------------------------------------------------------------------------------------------------------------------------------------------
#-------------------------------------------------------------- FFD -----------------------------------------------------------------------------
verbfirstfix<-subset(dataverb, dataverb$FFD!=0)
v1ffd<-lmer(log(FFD)~freq_ant+freq_int+MEant+MEint+IntAnt+(1+MEant+MEint+IntAnt|subject)+(1+MEant+MEint+IntAnt|Item), data=verbfirstfix, control=lmerControl(optimizer="bobyqa"))  
summary(v1ffd)
v2ffd<-lmer(log(FFD)~freq_ant+freq_int+MEant+gInt+uInt+(1+MEant+gInt+uInt|subject)+(1+MEant+gInt+uInt|Item), data=verbfirstfix, control=lmerControl(optimizer="bobyqa"))
summary(v2ffd)

#-------------------------------------------------------------- FPRT ------------------------------------------------------------------------------
verbfirstpass<-subset(dataverb, dataverb$FPRT!=0 & FFP==1)
v1fprt<-lmer(log(FPRT)~freq_ant+freq_int+MEant+MEint+IntAnt+(1|subject)+(1|Item), data=verbfirstpass, control=lmerControl(optimizer="bobyqa"))  
summary(v1fprt)
v2fprt<-lmer(log(FPRT)~freq_ant+freq_int+MEant+gInt+uInt+(1|subject)+(1|Item), data=verbfirstpass, control=lmerControl(optimizer="bobyqa")) 
summary(v2fprt)

#-------------------------------------------------------------- RBRT ------------------------------------------------------------------------------
verbrbound<-subset(dataverb, dataverb$RBRT!=0)
v1rbrt<-lmer(log(RBRT)~freq_ant+freq_int+MEant+MEint+IntAnt+(1|subject)+(1+MEant+MEint+IntAnt|Item), data=verbrbound, control=lmerControl(optimizer="bobyqa")) 
summary(v1rbrt)
v2rbrt<-lmer(log(RBRT)~freq_ant+freq_int+MEant+gInt+uInt+(1|subject)+(1+MEant+gInt+uInt|Item), data=verbrbound, control=lmerControl(optimizer="bobyqa"))   
summary(v2rbrt)

 #-------------------------------------------------------------- RPD ------------------------------------------------------------------------------
verbregpath<-subset(dataverb, dataverb$RPD!=0)
v1rpd<-lmer(log(RPD)~freq_ant+freq_int+MEant+MEint+IntAnt+(1+MEant+MEint+IntAnt|subject)+(1+MEant+MEint+IntAnt|Item), data=verbregpath, control=lmerControl(optimizer="bobyqa"))  
summary(v1rpd)
v2rpd<-lmer(log(RPD)~freq_ant+freq_int+MEant+gInt+uInt+(1+MEant+gInt+uInt|subject)+(1+MEant+gInt+uInt|Item), data=verbregpath, control=lmerControl(optimizer="bobyqa"))    
summary(v2rpd)

#-------------------------------------------------------------- RRT ------------------------------------------------------------------------------
verbreread<-subset(dataverb, dataverb$RRT!=0)
v1rrt<-lmer(log(RRT)~freq_ant+freq_int+MEant+MEint+IntAnt+(1+MEant+MEint+IntAnt|subject)+(1|Item), data=verbreread, control=lmerControl(optimizer="bobyqa")) 
summary(v1rrt)
v2rrt<-lmer(log(RRT)~freq_ant+freq_int+MEant+gInt+uInt+(1+MEant+gInt+uInt|subject)+(1|Item), data=verbreread, control=lmerControl(optimizer="bobyqa"))  
summary(v2rrt)

#-------------------------------------------------------------- TFT ------------------------------------------------------------------------------
verbtotalfix<-subset(dataverb, dataverb$TFT!=0)
v1tft<-lmer(log(TFT)~freq_ant+freq_int+MEant+MEint+IntAnt+(1|subject)+(1|Item), data=verbtotalfix, control=lmerControl(optimizer="bobyqa"))  
summary(v1tft)
v2tft<-lmer(log(TFT)~freq_ant+freq_int+MEant+gInt+uInt+(1|subject)+(1|Item), data=verbtotalfix, control=lmerControl(optimizer="bobyqa"))  
summary(v2tft)

#-------------------------------------------------------------- FPRP ------------------------------------------------------------------------------
v1regp<-glmer(RegP~freq_ant+freq_int+MEant+MEint+IntAnt+(1+MEant+MEint+IntAnt|subject)+(1+MEant+MEint+IntAnt|Item), data=dataverb, family=binomial(), control=glmerControl(optimizer="bobyqa"))
summary(v1regp)
v2regp<-glmer(RegP~freq_ant+freq_int+MEant+gInt+uInt+(1+MEant+gInt+uInt|subject)+(1+MEant+gInt+uInt|Item), data=dataverb, family=binomial(), control=glmerControl(optimizer="bobyqa"))
summary(v2regp)









#############################################################################
# 				Tables with Condition Means with Standard Errors 
##############################################################################

# define function for standard error for continuous DVs or proportions (here only used for regression probabilities; SEs of continuous variables computed with Morey-factor)
compute_se <- function(x){
	if (length(unique(x[!is.na(x)])>2) ){
		n <- length(x[!is.na(x)])
		sd(x, na.rm=TRUE) / sqrt(n)
	} else {
		m <- mean(x, na.rm=TRUE)
		n <- length(x[!is.na(x)])
		sqrt((m * (1 - m)) / n)
	}
}

# Function for SE with Morey correction factor:
# DV: vector containing dependent variable
# mf: Moray factor
SE <- function(DV,  MF){MF*sd(DV)/sqrt(length(DV))}
# set Morey factor for the present experiment:
mf <- sqrt(4/3) # Morey factor; because number of within-subj conditions = 4

#------------------------------------------------------------------------------------------
# Table of condition means and standard errors at the reflexive:
FFD<-t(round(tapply(zijifirstfix$FFD, zijifirstfix$condition, mean)))
FFD_SE<-t(round(tapply(zijifirstfix$FFD, zijifirstfix$condition, SE, mf)))
for (i in 1:4){ # for all conditions
	FFD[i]<-paste(FFD[i], FFD_SE[i], sep="(")
	FFD[i]<-paste(FFD[i], ")", sep="")
}
rownames(FFD)<-c("First-fixation")

FPRT<-t( round(tapply(zijifirstpass$FPRT, zijifirstpass$condition, mean)))
FPRT_SE<-t(round(tapply(zijifirstpass$FPRT, zijifirstpass$condition, SE, mf)))
for (i in 1:4){ # for all conditions
	FPRT[i]<-paste(FPRT[i], FPRT_SE[i], sep="(")
	FPRT[i]<-paste(FPRT[i], ")", sep="")
}
rownames(FPRT)<-c("First-pass")
RBRT<-t(round(tapply(zijirbound$RBRT, zijirbound$condition, mean)))
RBRT_SE<-t(round(tapply(zijirbound$RBRT, zijirbound$condition, SE, mf)))
for (i in 1:4){ # for all conditions
	RBRT[i]<-paste(RBRT[i], RBRT_SE[i], sep="(")
	RBRT[i]<-paste(RBRT[i], ")", sep="")
}
rownames(RBRT)<-c("Right-bounded")
RPD<- t(round(tapply(zijiregpath$RPD, zijiregpath$condition, mean)))
RPD_SE<-t(round(tapply(zijiregpath$RPD, zijiregpath$condition, SE, mf)))
for (i in 1:4){ # for all conditions
	RPD[i]<-paste(RPD[i], RPD_SE[i], sep="(")
	RPD[i]<-paste(RPD[i], ")", sep="")
}
rownames(RPD)<-c("Regression-path")
RRT<- t(round(tapply(zijireread$RRT, zijireread$condition, mean)))
RRT_SE<-t(round(tapply(zijireread$RRT, zijireread$condition, SE, mf)))
for (i in 1:4){ # for all conditions
	RRT[i]<-paste(RRT[i], RRT_SE[i], sep="(")
	RRT[i]<-paste(RRT[i], ")", sep="")
}
rownames(RRT)<-c("Re-reading")
TFT<- t(round(tapply(zijitotalfix$TFT, zijitotalfix$condition, mean)))
TFT_SE<-t(round(tapply(zijitotalfix$TFT, zijitotalfix$condition, SE, mf)))
for (i in 1:4){ # for all conditions
	TFT[i]<-paste(TFT[i], TFT_SE[i], sep="(")
	TFT[i]<-paste(TFT[i], ")", sep="")
}
rownames(TFT)<-c("Total-fixation")
RegP<- t(round(tapply(dataziji$RegP, dataziji$condition, mean),2))
RegP_SE<-t(round(tapply(dataziji$RegP, dataziji$condition, compute_se),2))
for (i in 1:4){ # for all conditions
	RegP[i]<-paste(RegP[i], RegP_SE[i], sep="(")
	RegP[i]<-paste(RegP[i], ")", sep="")
}
rownames(RegP)<-c("P(First-pass regression)")

meansZiji<-rbind(FFD, FPRT, RBRT,RPD,RegP, TFT, RRT)

print(meansZiji)

#------------------------------------------------------------------------------------------
# Table of condition means and standard errors at the post-critical region:
FFD<-t(round(tapply(spillfirstfix$FFD, spillfirstfix$condition, mean)))
FFD_SE<-t(round(tapply(spillfirstfix$FFD, spillfirstfix$condition, SE, mf)))
for (i in 1:4){ # for all conditions
	FFD[i]<-paste(FFD[i], FFD_SE[i], sep="(")
	FFD[i]<-paste(FFD[i], ")", sep="")
}
rownames(FFD)<-c("First-fixation")
FPRT<-t( round(tapply(spillfirstpass$FPRT, spillfirstpass$condition, mean)))
FPRT_SE<-t(round(tapply(spillfirstpass$FPRT, spillfirstpass$condition, SE, mf)))
for (i in 1:4){ # for all conditions
	FPRT[i]<-paste(FPRT[i], FPRT_SE[i], sep="(")
	FPRT[i]<-paste(FPRT[i], ")", sep="")
}
rownames(FPRT)<-c("First-pass")
RBRT<-t(round(tapply(spillrbound$RBRT, spillrbound$condition, mean)))
RBRT_SE<-t(round(tapply(spillrbound$RBRT, spillrbound$condition, SE, mf)))
for (i in 1:4){ # for all conditions
	RBRT[i]<-paste(RBRT[i], RBRT_SE[i], sep="(")
	RBRT[i]<-paste(RBRT[i], ")", sep="")
}
rownames(RBRT)<-c("Right-bounded")
RPD<- t(round(tapply(spillregpath$RPD, spillregpath$condition, mean)))
RPD_SE<-t(round(tapply(spillregpath$RPD, spillregpath$condition, SE, mf)))
for (i in 1:4){ # for all conditions
	RPD[i]<-paste(RPD[i], RPD_SE[i], sep="(")
	RPD[i]<-paste(RPD[i], ")", sep="")
}
rownames(RPD)<-c("Regression-path")
RRT<- t(round(tapply(spillreread$RRT, spillreread$condition, mean)))
RRT_SE<-t(round(tapply(spillreread$RRT, spillreread$condition, SE, mf)))
for (i in 1:4){ # for all conditions
	RRT[i]<-paste(RRT[i], RRT_SE[i], sep="(")
	RRT[i]<-paste(RRT[i], ")", sep="")
}
rownames(RRT)<-c("Re-reading")
TFT<- t(round(tapply(spilltotalfix$TFT, spilltotalfix$condition, mean)))
TFT_SE<-t(round(tapply(spilltotalfix$TFT, spilltotalfix$condition, SE, mf)))
for (i in 1:4){ # for all conditions
	TFT[i]<-paste(TFT[i], TFT_SE[i], sep="(")
	TFT[i]<-paste(TFT[i], ")", sep="")
}
rownames(TFT)<-c("Total-fixation")
RegP<- t(round(tapply(dataspill$RegP, dataspill$condition, mean),2))
RegP_SE<-t(round(tapply(dataspill$RegP, dataspill$condition, compute_se),2))
for (i in 1:4){ # for all conditions
	RegP[i]<-paste(RegP[i], RegP_SE[i], sep="(")
	RegP[i]<-paste(RegP[i], ")", sep="")
}
rownames(RegP)<-c("P(First-pass regression)")

meansSpill<-rbind(FFD, FPRT, RBRT,RPD,RegP, TFT, RRT)
print(meansSpill)

#------------------------------------------------------------------------------------------
# Table of condition means and standard errors at the pre-critical regio (verb):
FFD<-t(round(tapply(verbfirstfix$FFD, verbfirstfix$condition, mean)))
FFD_SE<-t(round(tapply(verbfirstfix$FFD, verbfirstfix$condition, SE, mf)))
for (i in 1:4){ # for all conditions
	FFD[i]<-paste(FFD[i], FFD_SE[i], sep="(")
	FFD[i]<-paste(FFD[i], ")", sep="")
}
rownames(FFD)<-c("First-fixation")
FPRT<-t( round(tapply(verbfirstpass$FPRT, verbfirstpass$condition, mean)))
FPRT_SE<-t(round(tapply(verbfirstpass$FPRT, verbfirstpass$condition, SE, mf)))
for (i in 1:4){ # for all conditions
	FPRT[i]<-paste(FPRT[i], FPRT_SE[i], sep="(")
	FPRT[i]<-paste(FPRT[i], ")", sep="")
}
rownames(FPRT)<-c("First-pass")
RBRT<-t(round(tapply(verbrbound$RBRT, verbrbound$condition, mean)))
RBRT_SE<-t(round(tapply(verbrbound$RBRT, verbrbound$condition, SE, mf)))
for (i in 1:4){ # for all conditions
	RBRT[i]<-paste(RBRT[i], RBRT_SE[i], sep="(")
	RBRT[i]<-paste(RBRT[i], ")", sep="")
}
rownames(RBRT)<-c("Right-bounded")
RPD<- t(round(tapply(verbregpath$RPD, verbregpath$condition, mean)))
RPD_SE<-t(round(tapply(verbregpath$RPD, verbregpath$condition, SE, mf)))
for (i in 1:4){ # for all conditions
	RPD[i]<-paste(RPD[i], RPD_SE[i], sep="(")
	RPD[i]<-paste(RPD[i], ")", sep="")
}
rownames(RPD)<-c("Regression-path")
RRT<- t(round(tapply(verbreread$RRT, verbreread$condition, mean)))
RRT_SE<-t(round(tapply(verbreread$RRT, verbreread$condition, SE, mf)))
for (i in 1:4){ # for all conditions
	RRT[i]<-paste(RRT[i], RRT_SE[i], sep="(")
	RRT[i]<-paste(RRT[i], ")", sep="")
}
rownames(RRT)<-c("Re-reading")
TFT<- t(round(tapply(verbtotalfix$TFT, verbtotalfix$condition, mean)))
TFT_SE<-t(round(tapply(verbtotalfix$TFT, verbtotalfix$condition, SE, mf)))
for (i in 1:4){ # for all conditions
	TFT[i]<-paste(TFT[i], TFT_SE[i], sep="(")
	TFT[i]<-paste(TFT[i], ")", sep="")
}
rownames(TFT)<-c("Total-fixation")
RegP<- t(round(tapply(dataverb$RegP, dataverb$condition, mean),2))
RegP_SE<-t(round(tapply(dataverb$RegP, dataverb$condition, compute_se),2))
for (i in 1:4){ # for all conditions
	RegP[i]<-paste(RegP[i], RegP_SE[i], sep="(")
	RegP[i]<-paste(RegP[i], ")", sep="")
}
rownames(RegP)<-c("P(First-pass regression)")

meansVerb<-rbind(FFD, FPRT, RBRT,RPD,RegP, TFT, RRT)
print(meansVerb)
